<?php

/*
Plugin Name: Portfolio
Plugin URI: http://www.wearepixel8.com
Description: A simple plugin that will create a portfolio custom post type with two custom taxonomies.
Version: 1.0.2
Author: We Are Pixel8
Author URI: http://www.wearepixel8.com
License:
	Copyright 2012 We Are Pixel8 <hello@wearepixel8.com>
	
	This program is free software; you can redistribute it and/or modify it under
	the terms of the GNU General Public License, version 2, as published by the Free
	Software Foundation.
	
	This program is distributed in the hope that it will be useful, but WITHOUT ANY
	WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
	PARTICULAR PURPOSE. See the GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

/*----------------------------------------------------------------------------*/
/* Register Services
/*----------------------------------------------------------------------------*/

add_action( 'init', 'wap8_register_services', 10 );

/**
 * Register Services
 *
 * Register services as a hierarchical custom taxonomy for the portfolio
 * custom post type.
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_register_services() {

	// declare $labels variable for storing labels array
	$labels = array(
		'name'                       => _x( 'Services', 'taxonomy general name', 'wap8theme-i18n' ),
		'singular_name'              => _x( 'Service', 'taxonomy singular name', 'wap8theme-i18n' ),
		'search_items'               => __( 'Search Services', 'wap8theme-i18n' ),
		'popular_items'              => __( 'Popular Services', 'wap8theme-i18n' ),
		'all_items'                  => __( 'All Services', 'wap8theme-i18n' ),
		'parent_item'                => __( 'Parent Service', 'wap8theme-i18n' ),
		'parent_item_colon'          => __( 'Parent Service:', 'wap8theme-i18n' ),
		'edit_item'                  => __( 'Edit Service', 'wap8theme-i18n' ),
		'update_item'                => __( 'Update Service', 'wap8theme-i18n' ),
		'add_new_item'               => __( 'Add New Service', 'wap8theme-i18n' ),
		'new_item_name'              => __( 'New Service', 'wap8theme-i18n' ),
		'separate_items_with_commas' => null,
		'add_or_remove_items'        => null,
		'choose_from_most_used'      => __( 'Choose from Most Used Services', 'wap8theme-i18n' )
	);
	
	// declare $args variable for storing args array
	$args = array(
		'label'             => __( 'Services', 'wap8theme-i18n' ),
		'labels'            => $labels,
		'public'            => true,
		'hierarchical'      => true,
		'show_ui'           => true,
		'show_in_nav_menus' => false,
		'args'              => array( 'orderby' => 'term_order' ),
		'rewrite'           => array( 'slug' => 'portfolio/services', 'with_front' => false ),
		'query_var'         => true
	);
	
	// register services as a custom taxonomy
	register_taxonomy(
		'services',  // unique handle to avoid potential conflicts
		'portfolio', // this custom taxonomy should only be associated with our custom post type
		$args        // array of arguments for this custom taxonomy
	);

}

/*----------------------------------------------------------------------------*/
/* Register Portfolio Tags
/*----------------------------------------------------------------------------*/

add_action( 'init', 'wap8_register_portfolio_tags', 10 );

/**
 * Register Portfolio Tags
 *
 * Register portfolio-tags as a non-hierarchical custom taxonomy for the
 * portfolio custom post type.
 *
 * @package Artisan Creatif.
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_register_portfolio_tags() {

	// declare $labels variable for storing labels array
	$labels = array(
		'name'                       => _x( 'Portfolio Tags', 'taxonomy general name', 'wap8theme-i18n' ),
		'singular_name'              => _x( 'Portfolio Tag', 'taxonomy singular name', 'wap8theme-i18n'),
		'search_items'               => __( 'Search Portfolio Tags', 'wap8theme-i18n' ),
		'popular_items'              => __( 'Popular Portfolio Tags', 'wap8theme-i18n' ),
		'all_items'                  => __( 'All Portfolio Tags', 'wap8theme-i18n' ),
		'edit_item'                  => __( 'Edit Portfolio Tag', 'wap8theme-i18n' ),
		'update_item'                => __( 'Update Portfolio Tag', 'wap8theme-i18n' ),
		'add_new_item'               => __( 'Add New Portfolio Tag', 'wap8theme-i18n' ),
		'new_item_name'              => __( 'New Portfolio Tag', 'wap8theme-i18n' ),
		'separate_items_with_commas' => __( 'Separate Portfolio Tags with commas', 'wap8theme-i18n' ),
		'add_or_remove_items'        => __( 'Add or Remove Portfolio Tags', 'wap8theme-i18n' ),
		'choose_from_most_used'      => __( 'Choose from Most Used Portfolio Tags', 'wap8theme-i18n' )
	);
	
	// declare $args variable for storing args array
	$args = array(
		'label'             => __( 'Portfolio Tags', 'wap8theme-i18n' ),
		'labels'            => $labels,
		'public'            => true,
		'hierarchical'      => false,
		'show_ui'           => true,
		'show_in_nav_menus' => false,
		'show_tagcloud'     => false,
		'args'              => array( 'orderby' => 'term_order' ),
		'rewrite'           => array( 'slug' => 'portfolio/portfolio-tags', 'with_front' => false ),
		'query_var'         => true
	);
	
	// register portfolio tags as a custom taxonomy
	register_taxonomy(
		'portfolio-tags', // unique handle to avoid potential conflicts
		'portfolio',      // this custom taxonomy should only be associated with our custom post type
		$args             // array of arguments for this custom taxonomy
	);

}

/*----------------------------------------------------------------------------*/
/* Portfolio
/*----------------------------------------------------------------------------*/

add_action( 'init', 'wap8_portfolio', 10 );

/**
 * Portfolio
 *
 * Register portfolio as a custom post type upon activation.
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio() {

	// declare $labels variable for storing labels array
	$labels = array(
		'name'               => _x( 'Portfolio', 'post type general name', 'wap8theme-i18n' ),
		'singular_name'      => _x( 'Portfolio', 'post type singular name', 'wap8theme-i18n' ),
		'add_new'            => _x( 'Add New', 'wap8-portfolio', 'wap8theme-i18n' ),
		'all_items'          => __( 'All Case Studies', 'wap8theme-i18n' ),
		'add_new_item'       => __( 'Add New Case Study', 'wap8theme-i18n' ),
		'edit'               => __( 'Edit', 'wap8theme-i18n' ),
		'edit_item'          => __( 'Edit Case Study', 'wap8theme-i18n' ),
		'new_item'           => __( 'New Case Study', 'wap8theme-i18n' ),
		'view'               => __( 'View', 'wap8theme-i18n' ),
		'view_item'          => __( 'View Case Study', 'wap8theme-i18n' ),
		'search_items'       => __( 'Search Portfolio', 'wap8theme-i18n' ),
		'not_found'          => __( 'No Case Studies found', 'wap8theme-i18n' ),
		'not_found_in_trash' => __( 'No Case Studies found in Trash', 'wap8theme-i18n' )
	);
	
	// declare $supports variable for storing supports array
	$supports = array(
		'title',
		'editor',
		'thumbnail',
		'excerpt',
		'revisions',
		'author'
	);
	
	// declare $args variable for storing args array
	$args = array(
		'labels'             => $labels,
		'public'             => true,
		'publicly_queryable' => true,
		'show_ui'            => true,
		'show_in_menu'       => true,
		'show_in_nav_menus'  => true,
		'query_var'          => true,
		'rewrite'            => array( 'slug' => 'portfolio', 'with_front' => false ),
		'capability_type'    => 'post',
		'hierarchical'       => false,
		'has_archive'        => false,
		'menu_position'      => 5,
		'menu_icon'          => get_template_directory_uri() . '/images/portfolio-icon.png',
		'supports'           => $supports
	);
	
	// register the post type
	register_post_type(
		'portfolio', // unique post type handle to avoid any potential conflicts
		$args        // array of arguments for this custom post type
	);

}

/*----------------------------------------------------------------------------*/
/* Updated Portfolio Messages
/*----------------------------------------------------------------------------*/
 
add_filter( 'post_updated_messages', 'wap8_updated_portfolio_messages', 10, 1 );

/**
 * Updated Portfolio Messages
 *
 * Customizing post updated messages for the wap8-portfolio custom post type.
 *
 * @param $messages Post updated messages
 * @return $messages Custom post updated messages
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_updated_portfolio_messages( $messages ) {

	global $post, $post_ID;

	$messages['portfolio'] = array(
		0  => '', // Unused. Messages start at index 1.
		1  => sprintf( __( 'Case study updated. <a href="%s">View case study</a>', 'wap8theme-i18n' ), esc_url( get_permalink( $post_ID ) ) ),
		2  => __( 'Custom field updated.', 'wap8theme-i18n' ),
		3  => __( 'Custom field deleted.', 'wap8theme-i18n' ),
		4  => __( 'Case study updated.', 'wap8theme-i18n' ),
		/* translators: %s: date and time of the revision */
		5  => isset( $_GET['revision'] ) ? sprintf( __( 'Case study restored to revision from %s', 'wap8theme-i18n' ), wp_post_revision_title( ( int ) $_GET['revision'], false ) ) : false,
		6  => sprintf( __( 'Case study published. <a href="%s">View case study</a>', 'wap8theme-i18n' ), esc_url( get_permalink( $post_ID ) ) ),
		7  => __( 'Case study saved.', 'wap8theme-i18n' ),
		8  => sprintf( __( 'Case study submitted. <a target="_blank" href="%s">Preview case study</a>', 'wap8theme-i18n' ), esc_url( add_query_arg( 'preview', 'true', get_permalink( $post_ID ) ) ) ),
		9  => sprintf( __( 'Case study scheduled for: <strong>%1$s</strong>. <a target="_blank" href="%2$s">Preview case study</a>', 'wap8theme-i18n' ),
		// translators: Publish box date format, see http://php.net/date
		date_i18n( __( 'M j, Y @ G:i', 'wap8lang' ), strtotime( $post->post_date ) ), esc_url( get_permalink( $post_ID ) ) ),
		10 => sprintf( __( 'Case study draft updated. <a target="_blank" href="%s">Preview case study</a>', 'wap8theme-i18n' ), esc_url( add_query_arg( 'preview', 'true', get_permalink( $post_ID ) ) ) ),
	);

	return $messages;

}

/*----------------------------------------------------------------------------*/
/* Portfolio Help Tabs Content
/*----------------------------------------------------------------------------*/

/**
 * Portfolio Help Tabs Content
 *
 * The content to be displayed in the portfolio help tabs.
 *
 * @param $tab Currently active tab
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio_help_tabs_content( $tab = 'wap8_portfolio_help_reminders' ) {
	
	if ( $tab == 'wap8_portfolio_help_reminders' ) { // if reminders tab is active
		
		ob_start(); // opening an output buffer to store the HTML content ?>
			
			<h2><?php _e( 'Things to Remember when Adding or Editing Portfolio Case Studies', 'wap8theme-i18n' ); ?></h2>
			
			<ul>
				<li><?php _e( 'Select a portfolio type. This is critical for correct rendering of the case study page.', 'wap8theme-i18n' ); ?></li>
				<li><?php _e( 'For stacked images or slider portfolio types, you do not have to insert images into the post. Upload images to the post and arrange to taste.', 'wap8theme-i18n' ); ?></li>
				<li><?php _e( 'Select at least one Service from your list of Services or add a new one. This is critical for the filtering functionality to work as intended.', 'wap8theme-i18n' ); ?></li>
				<li><?php _e( 'Select an image to be used as your featured image.', 'wap8theme-i18n' ); ?></li>
				<li><?php _e( 'Case studies you would like to appear in the large home page slider should be tagged, <strong>featured portfolio</strong>.', 'wap8theme-i18n' ); ?></li>
				<li><?php _e( 'Case studies you would like to appear in the carousel home page slider should be tagged, <strong>home portfolio</strong>.', 'wap8theme-i18n' ); ?></li>
			</ul>
			
			
		<?php
		
		return ob_get_clean(); // return the stored HTML content
	
	}
	
}

/*----------------------------------------------------------------------------*/
/* Portfolio Help Tabs
/*----------------------------------------------------------------------------*/

add_action( 'load-post.php', 'wap8_portfolio_help_tabs', 10 );
add_action( 'load-post-new.php', 'wap8_portfolio_help_tabs', 10 );

/**
 * Portfolio Help Tabs
 *
 * Add help tabs to the portfolio post editor screen.
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio_help_tabs() {
	
	$screen = get_current_screen();
    
    if ( $screen->id != 'portfolio' ) return; // if this is not the editor screen for portfolio posts, do nothing
    
    $screen->add_help_tab( array(
    	'id'      => 'wap8_portfolio_help_reminders',
		'title'   => __( 'Helpful Reminders', 'wap8theme-i18n' ),
		'content' => wap8_portfolio_help_tabs_content( 'wap8_portfolio_help_reminders' )
		)
	);
	
}

/*----------------------------------------------------------------------------*/
/* Custom Portfolio Columns
/*----------------------------------------------------------------------------*/

add_filter( 'manage_edit-portfolio_columns', 'wap8_custom_portfolio_columns', 10, 1 );

/**
 * Custom Portfolio Columns
 *
 * Customizing the columns for the wap8-portfolio custom post type edit screen.
 *
 * @param $columns Post columns
 * @return $columns Custom post columns
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_custom_portfolio_columns( $columns ) {

	$columns = array(
		'cb'                         => '<input type="checkbox" />',
		'title'                      => _x( __( 'Case Study', 'wap8theme-i18n' ), 'column name' ),
		'wap8-client-column'         => __( 'Client', 'wap8theme-i18n' ),
		'author'                     => __( 'Author', 'wap8theme-i18n' ),
		'wap8-services-column'       => __( 'Services', 'wap8theme-i18n' ),	// custom column for services
		'wap8-portfolio-tags-column' => __( 'Portfolio Tags', 'wap8theme-i18n' ), // custom column for portfolio tags
		'date'                       => _x( __( 'Date', 'wap8theme-i18n' ), 'column name' )
	);
	
	return $columns;

}	

/*----------------------------------------------------------------------------*/
/* Portfolio Columns Content
/*----------------------------------------------------------------------------*/

add_action( 'manage_portfolio_posts_custom_column', 'wap8_portfolio_columns_content', 10, 2 );

/**
 * Portfolio Columns Content
 *
 * Adding the custom taxonomies and client names to their respective custom
 * columns. The taxonomies should be comma separated anchors similar to post
 * categories and tags behavior.
 *
 * @param $column Custom columns
 * @param $post_id Post ID
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio_columns_content( $column, $post_id ) {
	
	global $post;
	
	switch ( $column ) {
		
		case 'wap8-client-column' : // client column
		
			$client = get_post_meta( $post->ID, 'wap8_client_name', true ); // get the client name from custom meta box
			
			if ( !empty( $client ) ) { // if a client name has been set
				
				echo $client;
				
			} else { // no client name has been set
				
				echo __( '<i>No client.</i>', 'wap8theme-i18n' );
				
			}
			
			break;
		
		case 'wap8-services-column' : // services column
			
			$terms = get_the_terms( $post_id, 'services' ); // get the services for the post

			if ( !empty( $terms ) ) { // if terms were found

				$out = array();

				foreach ( $terms as $term ) { // loop through each term, linking to the 'edit posts' page for the specific term
					$out[] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							add_query_arg(
								array(
									'post_type' => $post->post_type,
									'services'  => $term->slug
								), 'edit.php' ) ),
							esc_html( sanitize_term_field( 'name', $term->name, $term->term_id, 'services', 'display' ) )
						);
				}

				echo join( ', ', $out ); // join the terms and separate with a coma
				
			}

			else { // if no terms were found, output a default message
				
				_e( '<i>No services.</i>', 'wap8theme-i18n' );
				
			}
		
			break;
		
		case 'wap8-portfolio-tags-column' : // portfolio tags column
		
			$terms = get_the_terms( $post_id, 'portfolio-tags' ); // get the portfolio tags for the post

			if ( !empty( $terms ) ) { // if terms were found

				$out = array();

				foreach ( $terms as $term ) { // loop through each term, linking to the 'edit posts' page for the specific term
					$out[] = sprintf(
						'<a href="%s">%s</a>',
						esc_url(
							add_query_arg(
								array(
									'post_type'      => $post->post_type,
									'portfolio-tags' => $term->slug
								), 'edit.php' ) ),
							esc_html( sanitize_term_field( 'name', $term->name, $term->term_id, 'portfolio-tags', 'display' ) )
						);
				}

				echo join( ', ', $out ); // join the terms and separate with a coma
				
			}

			else { // if no terms were found, output a default message
				
				_e( '<i>No portfolio tags.</i>', 'wap8theme-i18n' );
				
			}
			
			break;
		
		default : // break out of the switch statement for everything else
		
			break;
		
	}

}

/*----------------------------------------------------------------------------*/
/* Portfolio Sortable Columns
/*----------------------------------------------------------------------------*/

add_filter( 'manage_edit-portfolio_sortable_columns', 'wap8_portfolio_sortable_columns', 10, 1 );

/**
 * Portfolio Sortable Columns
 *
 * Let WordPress know the client column should be sortable.
 *
 * @param $columns Post columns
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio_sortable_columns( $columns ) {

	$columns['wap8-client-column'] = 'wap8-client-column';

	return $columns;

}

/*----------------------------------------------------------------------------*/
/* Portfolio Edit Load
/*----------------------------------------------------------------------------*/

add_action( 'load-edit.php', 'wap8_portfolio_edit_load', 10 );

/**
 * Portfolio Edit Load
 *
 * Using the load-edit hook to insure we are on the edit.php screen. If so, add
 * our custom filter to request.
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_portfolio_edit_load() {
	
	add_filter( 'request', 'wap8_sort_portfolio_clients', 10, 1 );
	
}

/*----------------------------------------------------------------------------*/
/* Sort Portfolio Clients
/*----------------------------------------------------------------------------*/

/**
 * Sort Portfolio Clients
 *
 * If we are sorting the client column, sort _wap8_client_name by meta_value.
 *
 * @param $vars
 *
 * @package Artisan Creatif
 * @version 1.0.0
 * @since 2.0.0
 * @author Erik Ford for We Are Pixel8 <@notdivisible>
 *
 */

function wap8_sort_portfolio_clients( $vars ) {
	
	if ( isset( $vars['post_type'] ) && 'portfolio' == $vars['post_type'] ) { // if we are viewing the portfolio post type
		
		if ( isset( $vars['orderby'] ) && 'wap8-client-column' == $vars['orderby'] ) { // if we are ordering by client
			
			$vars =
			
				array_merge(
					$vars,
						array(
							'meta_key' => 'wap8_client_name',
							'orderby'  => 'meta_value'
					)
				);
			
		}
		
	}
	
	return $vars;
	
}

?>
